/**
 * A central script to handle various SweetAlert confirmations.
 * This script is designed to be reusable for any button with a 'confirm-action' class.
 */
document.addEventListener('DOMContentLoaded', function() {
    // Check if the required SMS balance variables exist and are numbers.
    const hasSmsCheck = typeof smsBalance !== 'undefined' && typeof smsThreshold !== 'undefined';
    const isSmsLow = hasSmsCheck && parseInt(smsBalance) <= parseInt(smsThreshold);
    
    // Define a cool-down period in milliseconds (e.g., 30 seconds).
    const COOL_DOWN_PERIOD = 30000;

    // Get all buttons that need a confirmation
    const confirmButtons = document.querySelectorAll('.confirm-action');

    // Add a click event listener to each button.
    confirmButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault(); // Prevent the default link behavior

            // Retrieve all necessary data attributes from the button
            const confirmUrl = this.getAttribute('href') || this.getAttribute('data-delete-url');
            const confirmTitle = this.getAttribute('data-title') || 'Are you sure?';
            const confirmText = this.getAttribute('data-text') || 'You are about to perform an action. This cannot be undone. Are you sure?';
            const confirmIcon = this.getAttribute('data-icon') || 'warning';
            const confirmButtonText = this.getAttribute('data-confirm-button-text') || 'Yes, proceed!';

            // --- Flow for buttons with SMS and Quick Action Checks ---
            if (button.classList.contains('sms-check-button')) {
                // Get timestamp for the quick action check
                const now = Date.now();
                // FIX: Use a unique key for each button based on its URL
                const lastClickKey = 'last-quick-action-' + btoa(confirmUrl); 
                const lastClick = localStorage.getItem(lastClickKey);
                
                // Check if the user is clicking too quickly
                if (lastClick && (now - lastClick < COOL_DOWN_PERIOD)) {
                    Swal.fire({
                        title: 'Slow down!',
                        text: 'You\'ve recently performed a similar action. Please wait a few moments before trying again.',
                        icon: 'info',
                        confirmButtonText: 'OK',
                    });
                    return; // Stop execution here
                }
                
                // Set the timestamp for the current click
                localStorage.setItem(lastClickKey, now);

                // Check for low SMS balance
                if (isSmsLow) {
                    Swal.fire({
                        title: 'Low SMS Balance!',
                        html: `Your current SMS balance is <b>${smsBalance}</b> and the threshold is <b>${smsThreshold}</b>. Topping up is highly recommended.`,
                        icon: 'warning',
                        confirmButtonText: 'Go to Settings',
                        showCancelButton: true,
                        cancelButtonText: 'Continue Anyway',
                    }).then((result) => {
                        if (result.isConfirmed) {
                            window.location.href = '<?= base_url("setting") ?>'; 
                        } else if (result.dismiss === Swal.DismissReason.cancel) {
                            showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
                        }
                    });
                } else {
                    // If SMS is not low, go straight to the main confirmation
                    showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
                }
            } else {
                // --- Default Flow for all other buttons ---
                showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
            }
        });
    });

    /**
     * Reusable function to show the main action confirmation SweetAlert.
     */
    function showMainConfirmation(url, title, text, icon, buttonText) {
        if (url) {
            Swal.fire({
                title: title,
                text: text,
                icon: icon,
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: buttonText,
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = url;
                }
            });
        }
    }
});
