/**
 * @fileoverview Centralized SweetAlert logic for consistent user actions.
 * This script handles confirmation dialogs, low balance warnings, and action cooldowns.
 */

// Define the cooldown duration in milliseconds (e.g., 60 seconds).
const COOLDOWN_DURATION = 60000;

document.addEventListener('DOMContentLoaded', function() {
    // Check if SMS balance variables are available.
    if (typeof smsBalance !== 'undefined' && typeof smsThreshold !== 'undefined') {
        // Select all elements with the 'confirm-action' class.
        const confirmActions = document.querySelectorAll('.confirm-action');

        // Add a click event listener to each of the selected elements.
        confirmActions.forEach(element => {
            element.addEventListener('click', function(e) {
                e.preventDefault(); // Stop the default link navigation.
                
                // Get the unique URL for this action to use as a key for the cooldown check.
                const actionUrl = this.getAttribute('href');

                // Check for cooldown period
                const lastActionTime = localStorage.getItem(`cooldown_${actionUrl}`);
                const now = Date.now();

                if (lastActionTime && (now - lastActionTime < COOLDOWN_DURATION)) {
                    const remainingTime = Math.ceil((COOLDOWN_DURATION - (now - lastActionTime)) / 1000);
                    Swal.fire({
                        title: 'Action on Cooldown',
                        text: `Please wait ${remainingTime} seconds before performing this action again.`,
                        icon: 'info',
                        confirmButtonText: 'OK',
                    });
                    return; // Exit the function to prevent further action.
                }

                const thresholdValue = parseInt(smsThreshold);

                // Check for low SMS balance based on the global threshold.
                if (smsBalance < thresholdValue) {
                    // Display the low balance warning alert.
                    Swal.fire({
                        title: 'Low SMS Balance',
                        text: `Your SMS balance (${smsBalance}) is low. This action requires a balance of at least ${thresholdValue}. Please top up to proceed.`,
                        icon: 'warning',
                        confirmButtonText: 'OK',
                    });
                    return; // Exit the function to prevent further action.
                }

                // Get URL and custom messages from the element's data attributes.
                const url = this.getAttribute('href');
                let title = this.dataset.title || 'Are you sure?';
                let text = this.dataset.text || 'This action cannot be undone.';
                const icon = this.dataset.icon || 'warning';
                const confirmButtonText = this.dataset.confirmButtonText || 'Yes, do it!';

                // Handle dynamic values in the text (e.g., student name).
                const dynamicValue = this.dataset.dynamicValue;
                if (dynamicValue) {
                    title = title.replace('{dynamic_value}', dynamicValue);
                    text = text.replace('{dynamic_value}', dynamicValue);
                }

                // Show the SweetAlert2 confirmation popup.
                Swal.fire({
                    title: title,
                    text: text,
                    icon: icon,
                    showCancelButton: true,
                    confirmButtonColor: '#3085d6',
                    cancelButtonColor: '#d33',
                    confirmButtonText: confirmButtonText,
                    customClass: {
                        popup: 'swal2-lg',
                    },
                }).then((result) => {
                    // If the user confirms, proceed to the URL.
                    if (result.isConfirmed) {
                        // Store the current timestamp in localStorage before navigating.
                        localStorage.setItem(`cooldown_${actionUrl}`, now.toString());
                        window.location.href = url;
                    }
                });
            });
        });
    }
});
