/**
 * @file sweetalert_helper.js
 *
 * This file contains a helper function for showing SweetAlert confirmations
 * with a check for SMS balance and to prevent double actions in quick succession.
 */

// We'll use a global variable to keep track of the last action's timestamp
// to prevent double-clicks.
const LAST_ACTION_TIMESTAMP_KEY = 'lastActionTimestamp';
const COOLDOWN_SECONDS = 60;

/**
 * Checks the SMS balance against a required threshold and shows a SweetAlert
 * confirmation dialog based on the result.
 */
document.addEventListener('DOMContentLoaded', function() {
    // We will use event delegation on the document body to handle clicks on any
    // .confirm-action element, even if it's added dynamically to the page.
    document.body.addEventListener('click', function(e) {
        // Find the closest ancestor with the .confirm-action class.
        const targetElement = e.target.closest('.confirm-action');

        if (targetElement) {
            // Get the required data from the clicked element's data attributes.
            const smsThreshold = parseFloat(targetElement.dataset.smsThreshold) || window.smsThreshold;
            const smsBalance = parseFloat(window.smsBalance);
            const href = targetElement.href;
            const title = targetElement.dataset.title || 'Are you sure?';
            const text = targetElement.dataset.text || 'This action cannot be undone.';
            const icon = targetElement.dataset.icon || 'warning';
            const confirmButtonText = targetElement.dataset.confirmButtonText || 'Yes, proceed!';

            // Check for the double-action cooldown.
            const lastActionTimestamp = localStorage.getItem(LAST_ACTION_TIMESTAMP_KEY);
            const currentTime = Date.now();
            if (lastActionTimestamp && (currentTime - lastActionTimestamp < COOLDOWN_SECONDS * 1000)) {
                e.preventDefault(); // Stop the default link or button action.
                Swal.fire({
                    title: 'Slow Down!',
                    text: 'An action was just performed. Please wait a moment before trying again.',
                    icon: 'info',
                    confirmButtonText: 'OK'
                });
                return;
            }

            // Check if the SMS balance is sufficient.
            if (smsBalance < smsThreshold) {
                e.preventDefault(); // Stop the default link or button action.
                Swal.fire({
                    title: 'SMS Balance Low!',
                    text: `Your SMS balance (${smsBalance}) is low. This action requires a balance of at least ${smsThreshold}. Please top up to proceed.`,
                    icon: 'warning',
                    confirmButtonText: 'OK'
                });
            } else {
                // If SMS balance is sufficient, show the standard confirmation dialog.
                e.preventDefault(); // Stop the default link or button action.
                Swal.fire({
                    title: title,
                    text: text,
                    icon: icon,
                    showCancelButton: true,
                    confirmButtonColor: '#3085d6',
                    cancelButtonColor: '#d33',
                    confirmButtonText: confirmButtonText
                }).then((result) => {
                    if (result.isConfirmed) {
                        // Store the timestamp of this action.
                        localStorage.setItem(LAST_ACTION_TIMESTAMP_KEY, Date.now());

                        // Check if the element is inside a form.
                        const form = targetElement.closest('form');
                        if (form) {
                            form.submit(); // Submit the form
                        } else if (href) {
                            window.location.href = href; // Redirect to the URL
                        }
                    }
                });
            }
        }
    });
});
