/**
 * Enhanced SweetAlert helper with SMS check and slowdown protection
 */
document.addEventListener('DOMContentLoaded', function() {
    // Get SMS data from PHP (injected by controller)
    const hasSmsCheck = typeof smsBalance !== 'undefined' && typeof smsThreshold !== 'undefined';
    const isSmsLow = hasSmsCheck && parseInt(smsBalance) <= parseInt(smsThreshold);
    
    // Cool-down period (30 seconds)
    const COOL_DOWN_PERIOD = 30000;

    // Add event listeners to all confirmation buttons
    const confirmButtons = document.querySelectorAll('.confirm-action');

    confirmButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();

            const confirmUrl = this.getAttribute('href') || this.getAttribute('data-delete-url');
            let confirmTitle = this.getAttribute('data-title') || 'Are you sure?';
            let confirmText = this.getAttribute('data-text') || 'You are about to perform an action.';
            const confirmIcon = this.getAttribute('data-icon') || 'warning';
            const confirmButtonText = this.getAttribute('data-confirm-button-text') || 'Yes, proceed!';

            // Replace variables in text from data attributes
            confirmText = confirmText.replace(/{(\w+)}/g, (match, p1) => {
                return this.getAttribute('data-' + p1) || match;
            });

            // --- SMS Check and Slowdown Protection ---
            if (button.classList.contains('sms-check-button')) {
                const now = Date.now();
                const lastClickKey = 'last-action-' + btoa(confirmUrl);
                const lastClick = localStorage.getItem(lastClickKey);
                
                // Slowdown check
                if (lastClick && (now - lastClick < COOL_DOWN_PERIOD)) {
                    Swal.fire({
                        title: 'Slow down!',
                        text: 'You\'ve recently performed a similar action. Please wait a few moments before trying again.',
                        icon: 'info',
                        confirmButtonText: 'OK',
                    });
                    return;
                }
                
                localStorage.setItem(lastClickKey, now);

                // SMS balance check
                if (isSmsLow) {
                    Swal.fire({
                        title: 'Low SMS Balance!',
                        html: `Your current SMS balance is <b>${smsBalance}</b> and the threshold is <b>${smsThreshold}</b>. Topping up is highly recommended.`,
                        icon: 'warning',
                        confirmButtonText: 'Go to Settings',
                        showCancelButton: true,
                        cancelButtonText: 'Continue Anyway',
                    }).then((result) => {
                        if (result.isConfirmed) {
                            window.location.href = smsSettingsUrl || '<?= base_url("setting") ?>';
                        } else if (result.dismiss === Swal.DismissReason.cancel) {
                            showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
                        }
                    });
                } else {
                    showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
                }
            } else {
                // For buttons without SMS check
                showMainConfirmation(confirmUrl, confirmTitle, confirmText, confirmIcon, confirmButtonText);
            }
        });
    });

    function showMainConfirmation(url, title, text, icon, buttonText) {
        Swal.fire({
            title: title,
            text: text,
            icon: icon,
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: buttonText,
            cancelButtonText: 'Cancel'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = url;
            }
        });
    }
});