/**
 * @fileoverview Centralized SweetAlert logic for consistent user actions.
 * This script handles confirmation dialogs and low balance warnings.
 */
document.addEventListener('DOMContentLoaded', function() {
    // Check if SMS balance variables are available.
    if (typeof smsBalance !== 'undefined' && typeof smsThreshold !== 'undefined') {
        const confirmActions = document.querySelectorAll('.confirm-action');

        confirmActions.forEach(element => {
            element.addEventListener('click', function(e) {
                e.preventDefault(); // Stop the link from navigating immediately.

                // Check for low SMS balance based on the global threshold.
                const thresholdValue = parseInt(smsThreshold);
                
                if (smsBalance < thresholdValue) {
                    // Display the low balance warning alert.
                    Swal.fire({
                        title: 'Low SMS Balance',
                        text: `Your SMS balance (${smsBalance}) is low. This action requires a balance of at least ${thresholdValue}. Please top up to proceed.`,
                        icon: 'warning',
                        confirmButtonText: 'OK',
                    });
                    return; // Exit the function to prevent further action.
                }

                // Get URL and custom messages from the element's data attributes.
                const url = this.getAttribute('href');
                let title = this.dataset.title || 'Are you sure?';
                let text = this.dataset.text || 'This action cannot be undone.';
                const icon = this.dataset.icon || 'warning';
                const confirmButtonText = this.dataset.confirmButtonText || 'Yes, do it!';

                // Handle dynamic values in the text (e.g., student name).
                const dynamicValue = this.dataset.dynamicValue;
                if (dynamicValue) {
                    title = title.replace('{dynamic_value}', dynamicValue);
                    text = text.replace('{dynamic_value}', dynamicValue);
                }

                // Show the SweetAlert2 confirmation popup.
                Swal.fire({
                    title: title,
                    text: text,
                    icon: icon,
                    showCancelButton: true,
                    confirmButtonColor: '#3085d6',
                    cancelButtonColor: '#d33',
                    confirmButtonText: confirmButtonText,
                    customClass: {
                        popup: 'swal2-lg',
                    },
                }).then((result) => {
                    // If the user confirms, proceed to the URL.
                    if (result.isConfirmed) {
                        window.location.href = url;
                    }
                });
            });
        });
    }
});
