<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Mpesa library
 *
 * Library with utilities to request Mpesa Payments using Daraja API
 */

class Mpesa_lib
{   
    private $CI;

  	public function __construct()
	{
		$this->CI =& get_instance();
	}

    public function init(array $config)
    {
        foreach ($config as $key => $value) {
            $this->CI->config->set_item("mpesa_{$key}", $value);
        }
    }

    public function send(
        $phone,
        $amount,
        $reference = "ACCOUNT",
        $description = "Transaction Description",
        $remark = "Remark",
        $callback = null
    ) {
        $phone = (substr($phone, 0, 1) == "+") ? str_replace("+", "", $phone) : $phone;
        $phone = (substr($phone, 0, 1) == "0") ? preg_replace("/^0/", "254", $phone) : $phone;
        $phone = (substr($phone, 0, 1) == "7") ? "254{$phone}" : $phone;

        $timestamp = date("YmdHis");
        $password  = base64_encode($this->CI->config->item('mpesa_shortcode') . $this->CI->config->item('mpesa_passkey') . $timestamp);

        $endpoint = ($this->CI->config->item('mpesa_env') == "live")
            ? "https://api.safaricom.co.ke/mpesa/stkpush/v1/processrequest"
            : "https://sandbox.safaricom.co.ke/mpesa/stkpush/v1/processrequest";

        $curl_post_data = array(
            "BusinessShortCode" => $this->CI->config->item('mpesa_shortcode'),
            "Password"          => $password,
            "Timestamp"         => $timestamp,
            "TransactionType"   => $this->CI->config->item('mpesa_shortcode_type'),
            "Amount"            => round($amount),
            "PartyA"            => $phone,
            "PartyB"            => ($this->CI->config->item('mpesa_shortcode_type') == 'CustomerPayBillOnline') ? $this->CI->config->item('mpesa_shortcode') : $this->CI->config->item('mpesa_headoffice'),
            "PhoneNumber"       => $phone,
            "CallBackURL"       => $this->CI->config->item('mpesa_callback_url'),
            "AccountReference"  => $reference,
            "TransactionDesc"   => $description,
            "Remark"            => $remark,
        );

        $response = $this->remote_post($endpoint, $curl_post_data);
        $result   = json_decode($response, true);

        return is_null($callback)
            ? $result
            : \call_user_func_array($callback, array($result));
    }
    
    public function register(
        $callback = null,
        $response_type = "Completed"
    ) {
        $endpoint = ($this->CI->config->item('mpesa_env') == "live")
            ? "https://api.safaricom.co.ke/mpesa/c2b/v2/registerurl"
            : "https://sandbox.safaricom.co.ke/mpesa/c2b/v2/registerurl";

        $curl_post_data = array(
            "ShortCode"       => $this->CI->config->item('mpesa_shortcode'),
            "ResponseType"    => $response_type,
            "ConfirmationURL" => $this->CI->config->item('mpesa_confirmation_url'),
            "ValidationURL"   => $this->CI->config->item('mpesa_validation_url'),
        );

        $response = $this->remote_post($endpoint, $curl_post_data);
        $result   = json_decode($response, true);

        return is_null($callback)
            ? $result
            : \call_user_func_array($callback, array($result));
    }

    public function remote_get($endpoint, $credentials = null)
    {
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array("Authorization: Basic " . $credentials));
        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);

        return curl_exec($curl);
    }

    /**
     * Perform a POST request to the M-PESA Daraja API
     * @param String $endpoint Daraja API URL Endpoint
     * @param Array $data Formated array of data to send
     *
     * @return string/bool
     */
    public function remote_post($endpoint, $data = array())
    {
        $token       = $this->token();
        $curl        = curl_init();
        $data_string = json_encode($data);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLOPT_URL, $endpoint);
        curl_setopt(
            $curl,
            CURLOPT_HTTPHEADER,
            array(
                "Content-Type:application/json",
                "Authorization:Bearer " . $token,
            )
        );

        return curl_exec($curl);
    }

    /**
     * Fetch Token To Authenticate Requests
     *
     * @return string Access token
     */
    public function token()
    {
        $endpoint = ($this->CI->config->item('mpesa_env') == "live")
            ? "https://api.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials"
            : "https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials";

        $credentials = base64_encode($this->CI->config->item('mpesa_key') . ":" . $this->CI->config->item('mpesa_secret'));
        $response    = $this->remote_get($endpoint, $credentials);
        $result      = json_decode($response);

        return isset($result->access_token) ? $result->access_token : "";
    }
}